/////////////////////////////////////////////////////////////////////////////////
//
// Shader  : TheEmu - Tunnel A.fsh
// Creator : TheEmu
// Version : 1.0
// Date    : 2015/05/12
// License : Creative Commons Attribution-NonCommercial-ShareAlike 3.0 Unported License.
//
/////////////////////////////////////////////////////////////////////////////////
//
// This shader was inspired by Inigo Quilez's Deform - Square Tunnel downloaded
// from ShaderToy.com,  but other than using the same Minkowski distance metric
// has nothing in common with it,  though  it  can be configured to produce the
// same tunnel shape.
//
// It is highly configurable, with the configuration parameters taking the form
// of a set of shader uniform inputs that may be defined in the scene file. The
// parameters control the tunnel shape, how its walls are tiled etc.
//
// With the single exception of the image that is used for the tunnel walls the
// control parameters are all optional  -  if they are not defined in the scene
// file they take the value 0 which is either the natural default value for the
// parameter or is treated as a special case indicating that some other default
// value is used.  All of the shader control parameters are described in detail
// in the following comments.
//
// Note, only the variables declared as uniform are inputs, the other variables 
// declared with them are simply constants from them.
//
/////////////////////////////////////////////////////////////////////////////////

// Standard shader inputs.

uniform float u_Elapsed;     // The elapsed time in seconds
uniform vec2  u_WindowSize;  // Window dimensions in pixels

   float wAspectRatio = u_WindowSize.x/u_WindowSize.y;

   vec2 wEqualizer = (wAspectRatio > 1.0) ? vec2( wAspectRatio, 1.0 )
                                       : vec2( 1.0, 1.0/wAspectRatio );

/////////////////////////////////////////////////////////////////////////////////

// Shader inputs that may be defined in the scene file.

// The image to be used for the tunnel walls is supplied using the following 2D
// sampler input channel. It may be any valid 2D sample,  e.g.  a  static image
// defined by a texture statement in the scene file, or a dynamic one generated
// in a framebuffer by a shader or it may be from a clip.  An image must always
// be provided.

uniform sampler2D iChannel0;

// By default, speed 0.0, the tunnel will be static.  Positive values for speed
// will generate the effect of motion into the tunnel and negative values  that
// of motion out of the tunnel, though using "unusual" values for other control
// parameters can reverse this.  If used speed should normally have an absolute
// value of less than 10, though higher speeds are permitted.

uniform float speed;

   float time = u_Elapsed * speed;

// The shader can operate in several modes, these being
//
// Symetric 180 degree wrapping modes
//
//  00 - Tunnel is symetric about vertical axis
//  01 - Tunnel is symetric about horizontal axis
//
//  10 - As mode 0 but flipped about horizontal axis
//  11 - As mode 1 but flipped about vertical axis
//
// 360 wrapping modes starting at multiples of 45 degrees
//
//  20 to 27 - anticlockwise, second digit specifies starting segment 
//  30 to 27 - clockwise, second digit specifies starting segment 
//

uniform int mode;

// The image used for the tunnel is repeated as many times as is  necessary  to
// cover the full 360 degrees of the tunnel. By default the repeat angle is set
// by the mode parameter being 180 for modes 0..3 and 360 for modes 11..13, but
// this is can be overridden by specifying a non zero value for the repeatAngle
// which is the angle in degrees after which the image is repeated.

uniform float repeatAngle;

// The tunnel's shape is controlled by the shape parameter. The first component
// of which controls the basic shape and the second its aspect  ratio.  If  the
// first component is 0.0 then the tunnel is eliptical, values greater than 0.0
// correspond to shapes intermediate between elipses and rectangles with  large
// values,  e.g. 10, produce tunnels that are rectangular with rounded corners.
// Negative values may also be used, but they are less useful.  The second part
// of shape controls the aspect ratio of the tunnel,  i.e.  its width to height
// ratio.  However, an aspect ratio of 0.0 is a special case which results in a
// tunnel having the same aspect ratio as the window in which the tunnel is  to
// be rendered.  Shape  defaults  to (0.0,0.0) resulting in an eliptical tunnel
// with the same aspect ratio as the window in which it is rendered.

uniform vec2 shape;

   float tunnelSectionShape = ( 1.0 + shape.x ) * 2.0;
   float tunnelAspectRatio  = (shape.y==0.0) ? wAspectRatio : shape.y;

// By default the tunnel's vanishing point will be the center of the window  in
// which it is being displayed, but this may be changed by spacifying that some
// other center is used.  The values for center take the form (dx,dy) where the
// dx component ranges from -1.0 at the left edge of the window to +1.0 at  the
// right edge and dy ranges from -1.0 at the bottom of the window  to  +1.0  at
// the top. Values outside of these ranges are valid but are not expected to be
// used as they would move the tunnel's center out of the window.

uniform vec2 center;

// By default the X and Y coordinates are horizontal and vertical, however they
// may be rotated by specifying a non zero rotation angle, alpha, in degrees. A
// positive value for alpha signifies a clockwise rotation of the tunnel.

uniform float alpha;

   float sinAlpha = sin(radians(alpha));
   float cosAlpha = cos(radians(alpha));

   mat2 rotateAlpha = mat2(cosAlpha,sinAlpha,-sinAlpha,cosAlpha);

// The generated tunnel will appear to be composed of  repetitions  of  tubular
// elements laid end to end along the Z axis, i.e. into the screen. The size of
// these elements may be controlled by specifying a non-zero value  for  zScale
// which will then be used to multiply the default size.  Any value may be used
// for zScale, though 0.0 is ignored (effectively being used as if it were 1.0)
// but typical values are expected to be between 0.5 and 5.0.

uniform float zScale;

// By default there is no fading as the distance along  the  tunnel  increases,
// however this can be overriden by specifying a non-zero value for  fade.  Any
// value can be used but typical values are expected to be in the range 0.0  to
// 5.0 as larger values will blank out most of the tunnel.

uniform float fade;

// By default any fading will be towards black but this can be overriden by the
// use of fadeColour. The three components of which are the red, green and blue
// intensities of the colour to fade towards.  Fading towards black will simply
// darken the tunnel while fading toward white makes the tunnel misty.

uniform vec3 fadeColour;

   vec4 fadeColourV4 = vec4(fadeColour,1.0);

// By default the opacity of the tunnel will match that of the  image  used  to
// provide the texture its walls. However, this may be overridden by specifying
// a non-zero value for translucency. Translucency values are clamped to lie in
// the range 0.0 to 1.0 with the tunnel walls becoming more transparent as  the
// value is increased  -  with 1.0 being fully transparent and hence invisible.
// Note,  however,  that any fully transparent region in the image used for the
// runnel walls texture will always remain fully transparent.

uniform float translucency;

   float opacity = 1.0 - clamp(translucency,0.0,1.0);

// By default each copy of the image used for the walls will be oriented in the
// same way.  Unless the image has been specialy designed to avoid it this will
// cause clearly visible discontinuities where the image tiles meet.  This  can
// be aleviated by flipping the alternate tiles and this is controlled  by  the
// tileScheme parameter.  This is a two digit integer where themost significant
// digit controls the tile flipping in the y direction,  i.e. verticaly,  while
// the least significant digit controls flipping in the z direction, i.e. along
// the tunnel. In each case the interpretation of the digit is
//
//     0 - no tile flipping
//     1 - flip even numbered tiles
//     2 - flip odd numbered tiles

uniform int tileScheme;

// By default the tunnel walls are tiled using a simple rectangular pattern  of
// repetitions of the tunnel wall image with one axis parallel to that  of  the
// of the tunnel.  However,  this pattern may be modified such that the pattern
// twists around the tunnel axis. This is controlled by the following parameter
// which works by changing the effective shape of the tunnel wall image from  a
// rectangles to a diamond using the transformation
//
//      (X,Y) => ( X + A*Y, Y + B*X )
// 
// where A and B are constants given by the two components of twist.
//
// Depending on the values of A and B a variety of effects can be obtained,  in
// the following it is assumed that a 360 degree wrapping mode is used.
//
// Non-zero values for A cause lines that were perpendicular to the axis of the
// tunnel,  i.e. the closed loops made by vertical sections through the tunnel,
// to spiral about the axis. In general this will introduce discontinuities and
// a closed loop will become a spiral segement, but if A*N is an integer, where
// N is 360/repeatAngle,  then the end of one segment will connect to the start
// of the next giving a continuous spiral down the tunnel.
//
// Non-zero values for B cause lines that were parallel to the tunnel's axis to
// spiral about that axis leaving lines at right angles to the axis  unchanged.
// In this case all values lead to continuous spirals.
//
// The simplest spirals occur if either A or B is zero.  If  both  are non-zero
// then more complex twisting of the pattern will result. A special case occurs
// when A and B are both 1.0 as the transformation given above the reduces to
//
//        (X,Y) => ( X + Y, X + Y )
//
// and instead of transforming the image to a diamond it becomes transformed to
// a single line. In this case the tiles are one dimensional and the end result
// is the tunnel walls are tiled with ribbons. 
//
// Values for A and B are expected to be in the range -5.0 to +0.5, values that
// lie outside this range are valid but the resulting spirals are rather tight.

uniform vec2 twist;

// By default the tunnel is endless,  however this can be changed by specifying
// a positive value for endSize.  Assuming that all other parameters have their
// default values setting endSize to 1.0 will cause the end of  the  tunnel  to
// just reach the edges of the window and smaller values cause it to reduce  in
// size, and hence retreat down the tunnel into the distance.

uniform float endSize;

// By default the end of the tunnel is transparent,  but any colour may be used
// by setting endColour. The components are the Red, Green and Blue intensities
// and the opacity. It will have no effect if the endsize is 0.0 or less.

uniform vec4 endColour;

// The tunnel can be regarded as a series of tubes aligned along the Z axis. By
// default there is no gap btween the tubes and the tunnel is continuous.  This
// may however be overrriden by providing a non zero value  for  gapSize  which
// specifies the relative size of the inter-tube gaps. Values for gapSize ought
// to lie between 0.0 and 1.0 corresponding to no gap and all gap respectively.

uniform float gapSize;

   float segmentSize = 1.0 - clamp(gapSize,0.0,1.0);

// By default the gaps in the tunnel is transparent, but any colour may be used
// by setting gapColour. The components are the Red, Green and Blue intensities
// and the opacity. It will have no effect if the gapsize is 0.0 or less.

uniform vec4 gapColour;

/////////////////////////////////////////////////////////////////////////////////

// TunnelCoordinates determines the (X,Y) coordinates of the current pixel. The
// coordinates origin is at the center of the tunnel.

vec2 TunnelCoordinates ( void )
 {

   // Rescale so that all the range of X and Y are both from -1 to +1.

   vec2 uv = ( 2.0*gl_FragCoord.xy/u_WindowSize.xy - 1.0 );

   // Move the origin to be the center of the tunnel.

   uv = uv - center;

   // Rescale so that the X and Y both have equal scales.

   uv = uv * wEqualizer;

   // Rotate by alpha.

   return rotateAlpha * uv;

 }

/////////////////////////////////////////////////////////////////////////////////

// TunnelTheta calculates the angle round the tunnel that  corresponds  to  the
// current pixel and rescales it such that 1.0 corresponds to the repeat  angle
// used for the tunnel wall tiling.  The modes all reduce to one of two general
// cases which are handled by their own functions.

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

// TunnelThetaA handles the simple 180 degree wrapping modes.

float TunnelThetaA ( vec2 uv, float a, float b )
 {
   return a*radians(180.0) + b*abs(atan(uv.y,uv.x));
 }

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

// TunnelThetaB handles the oblique 180 degree wrapping modes.

float TunnelThetaB ( vec2 uv, float a, float b )
 {
   float theta = atan(a*uv.y,a*uv.x) + b*radians(360.0+45.0);
   theta = mod(theta,radians(360.0));
   if ( theta > radians(180.0) ) theta = theta - radians(360.0);
   return abs(theta);
 }

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

// TunnelThetaC handles the 360 degree wrapping modes.

float TunnelThetaC ( vec2 uv, int n, float a )
 {
   return radians(45.0*n) + a*atan(uv.y,uv.x);
 }

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

float TunnelTheta ( vec2 uv )
 {
   float theta;

   // Determine which f the mode handlers to call and then call it
   // with the appropriate set mode specific parameter values.

   int m = int(mode);
   if ( m > 19 ) m = m - int(mod(m,10));

   switch ( m )
    {
      case 00 : theta = TunnelThetaA ( uv.yx,  0.0, +1.0 ); break;
      case 01 : theta = TunnelThetaA ( uv.xy,  0.0, +1.0 ); break;
      case 10 : theta = TunnelThetaA ( uv.yx, +1.0, -1.0 ); break;
      case 11 : theta = TunnelThetaA ( uv.xy, +1.0, -1.0 ); break;

      case 02 : theta = TunnelThetaB ( uv.yx, +1.0, -1.0 ); break;
      case 03 : theta = TunnelThetaB ( uv.xy, -1.0, +1.0 ); break;
      case 12 : theta = TunnelThetaB ( uv.xy, +1.0, +1.0 ); break;
      case 13 : theta = TunnelThetaB ( uv.yx, -1.0, -1.0 ); break;

      case 20 : theta = TunnelThetaC ( uv, mode-20, +1.0 ); break;
      case 30 : theta = TunnelThetaC ( uv, mode-30, -1.0 ); break;
    }

   // Get the wall tiling repeat angle replacing any defaulted, i.e
   // zero, value with the appropriate mode dependant default.

   float rptAngle = repeatAngle;

   if ( rptAngle == 0.0 )
    {  rptAngle = (mode<20) ? 180.0 : 360.0;
    }

   // Return the rescaled theta angle.

   return theta/radians(rptAngle);

 }

/////////////////////////////////////////////////////////////////////////////////

// Tunnel radius calculates the on screen radius of the tunnel coresponding  to
// the current pixel, i.e. its distance from the center of the tunnel.  However
// a modified metric is used for the radius which has the effect of  distorting
// the tunnel according the shape parameter.
float TunnelRadius ( vec2 uv )
 {
   vec2 qq = uv * vec2(1.0,tunnelAspectRatio);

   qq = abs(qq);

   qq.x = pow ( qq.x, tunnelSectionShape );
   qq.y = pow ( qq.y, tunnelSectionShape );

   return pow ( qq.x + qq.y, 1.0/tunnelSectionShape );

 }


/////////////////////////////////////////////////////////////////////////////////

// TunnelWall determines the colour, including the alpha component, to be used
// for the current tunnel wall pixel.

vec4 TunnelWall ( float r, float theta )
 {
   vec2 qq = vec2 ( 1.0/r + time, theta );

   qq = qq + twist * qq.yx;

   // Optionaly adjust the effective z axis scaling, i.e. that for
   // distances measured into the tunnel.

   if ( zScale != 0.0 ) qq.x = qq.x / zScale;

   // Optionaly flip the texture indices to eliminate first  order
   // discontinuities between neighbouring tunnel patches. The two
   // digit integer tileScheme uniform controls this process.

   int kx = int(floor(qq.x)); kx = kx - 2*(kx/2);
   int ky = int(floor(qq.y)); ky = ky - 2*(ky/2);

   switch ( tileScheme )
    { 
      case 00: break;

      case 01: if ( kx == 0 ) qq.x = 1.0 - fract(qq.x); break;

      case 02: if ( kx != 0 ) qq.x = 1.0 - fract(qq.x); break;

      case 10: if ( ky == 0 ) qq.y = 1.0 - fract(qq.y); break;

      case 11: if ( ky == 0 ) qq.y = 1.0 - fract(qq.y);
               if ( kx == 0 ) qq.x = 1.0 - fract(qq.x); break;

      case 12: if ( ky == 0 ) qq.y = 1.0 - fract(qq.y);
               if ( kx != 0 ) qq.x = 1.0 - fract(qq.x); break;

      case 20: if ( ky != 0 ) qq.y = 1.0 - fract(qq.y); break;

      case 21: if ( ky != 0 ) qq.y = 1.0 - fract(qq.y);
               if ( kx == 0 ) qq.x = 1.0 - fract(qq.x); break;

      case 22: if ( ky != 0 ) qq.y = 1.0 - fract(qq.y);
               if ( kx != 0 ) qq.x = 1.0 - fract(qq.x); break;
    }

   qq = fract(qq);

   // Get the pixel's colour and apply translucency.  Note that any
   // transparent region in the tunnel's source texture will remain
   // transparent thereby allowing any  background  to  be  visible 
   // through the windows that they provide.

   vec4 colour;

   if ( qq.x > segmentSize )
    {
      colour = gapColour;
    }
   else
    {
      qq.x = qq.x / segmentSize;
      colour = texture2D ( iChannel0, qq );
      colour.a = colour.a * opacity;
     }

   // Apply any fading.

   return mix ( fadeColourV4, colour, clamp(pow(r,fade),0.0,1.0) );

 }

/////////////////////////////////////////////////////////////////////////////////

void main( void )
 {
   vec2 uv = TunnelCoordinates();
   float r = TunnelRadius ( uv );

   gl_FragColor = (r<endSize) ? endColour
                              : TunnelWall ( r, TunnelTheta(uv) );

 }

/////////////////////////////////////////////////////////////////////////////////
